/**
  ******************************************************************************
 * @file    hal_adc.h
 * @brief   Header file of ADC Module library.
 *
 * @version V1.0
 * @date    21 Aug 2018
 * @author  AE Team
 * @note
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 ******************************************************************************
 */


#ifndef __HAL_ADC_H__
#define __HAL_ADC_H__

#include <hic.h>
#include "type.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @addtogroup ADC
  * @{
  */

/**
  * @defgroup Adc_Private_Types adc private types
  * @{
  */

/**
  * @brief ADC speed mode choose
  */
typedef enum {
	ADC_LOW_SPEED  = 0x0,	/* Adc low speed mode */
	ADC_HIGH_SPEED = 0x1,	/* Adc high speed mode */
} adc_speed_t;

/**
  * @brief ADC sampling time
  */
typedef enum {
	ADC_TADCLK_2  = 0x0,
	ADC_TADCLK_4  = 0x1,
	ADC_TADCLK_8  = 0x2,
	ADC_TADCLK_16 = 0x3,
} adc_samp_clk_t;

/**
  * @brief ADC convert frequence choose
  */
typedef enum {
	ADC_CLKDIV_1  = 0x0,
	ADC_CLKDIV_2  = 0x1,
	ADC_CLKDIV_4  = 0x2,
	ADC_CLKDIV_8  = 0x3,
	ADC_CLKDIV_16 = 0x4,
	ADC_CLKDIV_32 = 0x5,
	ADC_CLKDIV_64 = 0x6,
	ADC_CLK_LRC   = 0x7,
} adc_clk_div_t;

/**
  * @brief ADC data align mode
  */
typedef enum {
	ADC_DATAALIGN_HIGH = 0x0,
	ADC_DATAALIGN_LOW  = 0x1,
} adc_align_t;

/**
  * @brief ADC start sampling mode
  */
typedef enum {
	ADC_SOFT_START = 0x0,
	ADC_HARD_START = 0x1,
} adc_start_t;

/**
  * @brief ADC convert channel choose
  */
typedef enum {
	ADC_CHANNEL_0 = 0x0,
	ADC_CHANNEL_1 = 0x1,
	ADC_CHANNEL_2 = 0x2,
	ADC_CHANNEL_3 = 0x3,
	ADC_CHANNEL_4 = 0x4,
	ADC_CHANNEL_5 = 0x5,
	ADC_CHANNEL_6 = 0x6,
	ADC_CHANNEL_7 = 0x7,
	ADC_CHANNEL_8 = 0x8,
	ADC_CHANNEL_9 = 0x9,
} adc_channel_t;

/**
  * @brief  ADC handle Structure definition
  */
typedef struct adc_init_s {
	adc_speed_t speed;
	adc_samp_clk_t samp_clk;
	adc_clk_div_t clk_div;
	adc_align_t align;
	adc_start_t start_mode;
	adc_channel_t channel;
} adc_init_t;
/**
  * @}
  */

#define ADC_START()			(ADTRG = 1)

/**
  * @defgroup Adc_Private_Micros adc Private Micros
  * @{
  */

#define IS_ADC_SPEED_TYPE(x)			(((x) == ADC_LOW_SPEED) || \
						 ((x) == ADC_HIGH_SPEED))
#define IS_ADC_SAMP_CLK_TYPE(x)			(((x) == ADC_TADCLK_2) || \
						 ((x) == ADC_TADCLK_4) || \
						 ((x) == ADC_TADCLK_8) || \
						 ((x) == ADC_TADCLK_16))
#define IS_ADC_CLK_DIV_TYPE(x)			(((x) == ADC_CLKDIV_1) || \
						 ((x) == ADC_CLKDIV_2) || \
						 ((x) == ADC_CLKDIV_4) || \
						 ((x) == ADC_CLKDIV_8) || \
						 ((x) == ADC_CLKDIV_16) || \
						 ((x) == ADC_CLKDIV_32) || \
						 ((x) == ADC_CLKDIV_64) || \
						 ((x) == ADC_CLK_LRC))
#define IS_ADC_ALIGN_TYPE(x)			(((x) == ADC_DATAALIGN_HIGH) || \
						 ((x) == ADC_DATAALIGN_LOW))
#define IS_ADC_START_TYPE(x)			(((x) == ADC_SOFT_START) || \
						 ((x) == ADC_HARD_START))						 
#define IS_ADC_CHANNLE_TYPE(x)			(((x) == ADC_CHANNEL_0) || \
						 ((x) == ADC_CHANNEL_1) || \
						 ((x) == ADC_CHANNEL_2) || \
						 ((x) == ADC_CHANNEL_3) || \
						 ((x) == ADC_CHANNEL_4) || \
						 ((x) == ADC_CHANNEL_5) || \
						 ((x) == ADC_CHANNEL_6) || \
						 ((x) == ADC_CHANNEL_7) || \
						 ((x) == ADC_CHANNEL_8) || \
						 ((x) == ADC_CHANNEL_9))
/**
  * @}
  */

/**
  * @addtogroup Adc_Public_Functions
  * @{
  */
						 
extern hal_status_t adc_init(adc_init_t *init);
extern uint16_t adc_get_value(void);
/**
  *@}
  */

/**
  * @}
  */

/**
  * @}
  */

#endif /* __HAL_ADC_H */
